<?php
// Secure session configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_samesite', 'Lax');

// Enable secure cookies on HTTPS
if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
    ini_set('session.cookie_secure', 1);
}

session_start();

// Regenerate session ID on login to prevent fixation
function regenerate_session() {
    if (session_status() === PHP_SESSION_ACTIVE) {
        session_regenerate_id(true);
    }
}

// Check if user is logged in
function logged_in() {
    return isset($_SESSION['user']);
}

// Redirect to login if not authenticated
function confirm_logged_in() {
    if (!logged_in()) {
        header("Location: ../auth/login.php");
        exit;
    }
}

// ============ CSRF Protection ============

// Generate CSRF token
function csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Output CSRF input field for forms
function csrf_input() {
    return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars(csrf_token(), ENT_QUOTES, 'UTF-8') . '">';
}

// Verify CSRF token
function csrf_verify($token) {
    if (!isset($_SESSION['csrf_token']) || !isset($token)) {
        return false;
    }
    return hash_equals($_SESSION['csrf_token'], $token);
}

// ============ Input Sanitization Helpers ============

// Sanitize string input
function get_str($key, $default = '') {
    if (!isset($_POST[$key]) && !isset($_GET[$key])) {
        return $default;
    }
    $value = $_POST[$key] ?? $_GET[$key];
    return is_string($value) ? trim($value) : $default;
}

// Get integer input
function get_int($key, $default = 0) {
    if (!isset($_POST[$key]) && !isset($_GET[$key])) {
        return $default;
    }
    $value = $_POST[$key] ?? $_GET[$key];
    return filter_var($value, FILTER_VALIDATE_INT, ['options' => ['default' => $default]]);
}

// Get URL input
function get_url($key, $default = '') {
    if (!isset($_POST[$key]) && !isset($_GET[$key])) {
        return $default;
    }
    $value = $_POST[$key] ?? $_GET[$key];
    $filtered = filter_var($value, FILTER_VALIDATE_URL);
    return $filtered !== false ? $filtered : $default;
}

// Get email input
function get_email($key, $default = '') {
    if (!isset($_POST[$key]) && !isset($_GET[$key])) {
        return $default;
    }
    $value = $_POST[$key] ?? $_GET[$key];
    $filtered = filter_var($value, FILTER_VALIDATE_EMAIL);
    return $filtered !== false ? $filtered : $default;
}

// Get boolean input (checkbox)
function get_bool($key, $default = false) {
    return isset($_POST[$key]) || isset($_GET[$key]);
}

// ============ Output Escaping ============

// HTML escape for safe output
function e($value) {
    if ($value === null) {
        return '';
    }
    return htmlspecialchars((string)$value, ENT_QUOTES, 'UTF-8');
}

// URL encode for safe URLs
function url_encode_safe($value) {
    return urlencode((string)$value);
}

// ============ Login Throttling ============

// Track failed login attempts
function track_failed_login($identifier) {
    $key = 'login_attempts_' . md5($identifier);
    if (!isset($_SESSION[$key])) {
        $_SESSION[$key] = ['count' => 0, 'time' => time()];
    }
    
    // Reset after 15 minutes
    if (time() - $_SESSION[$key]['time'] > 900) {
        $_SESSION[$key] = ['count' => 0, 'time' => time()];
    }
    
    $_SESSION[$key]['count']++;
    $_SESSION[$key]['time'] = time();
}

// Check if login is throttled
function is_login_throttled($identifier) {
    $key = 'login_attempts_' . md5($identifier);
    if (!isset($_SESSION[$key])) {
        return false;
    }
    
    // Reset after 15 minutes
    if (time() - $_SESSION[$key]['time'] > 900) {
        unset($_SESSION[$key]);
        return false;
    }
    
    // Block after 5 failed attempts
    return $_SESSION[$key]['count'] >= 5;
}

// Clear login attempts on success
function clear_login_attempts($identifier) {
    $key = 'login_attempts_' . md5($identifier);
    unset($_SESSION[$key]);
}

// ============ Security Headers ============

// Set security headers
function set_security_headers() {
    header("X-Content-Type-Options: nosniff");
    header("X-Frame-Options: SAMEORIGIN");
    header("X-XSS-Protection: 1; mode=block");
    header("Referrer-Policy: strict-origin-when-cross-origin");
    
    // Content Security Policy
    $csp = "default-src 'self'; ";
    $csp .= "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.jsdelivr.net https://fonts.googleapis.com; ";
    $csp .= "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com; ";
    $csp .= "img-src 'self' data: https: http:; ";
    $csp .= "font-src 'self' https://fonts.gstatic.com; ";
    $csp .= "connect-src 'self' https://onesignal.com;";
    
    header("Content-Security-Policy: " . $csp);
}

// Apply security headers (call this in header.php)
set_security_headers();
?>